---

## 📄 File Contents

### `/install.php`
```php
<?php
/**
 * Billing System Installer for cPanel
 * Web-based installation script
 */

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

class BillingInstaller {
    
    private $steps = [
        1 => 'System Requirements',
        2 => 'Database Configuration',
        3 => 'Admin Account Setup',
        4 => 'Site Configuration',
        5 => 'Payment Gateway Setup',
        6 => 'Installation Complete'
    ];
    
    private $currentStep;
    private $errors = [];
    private $success = [];
    
    public function __construct() {
        $this->currentStep = isset($_GET['step']) ? (int)$_GET['step'] : 1;
    }
    
    public function run() {
        // Check if already installed
        if (file_exists('config/installed.lock') && $this->currentStep !== 6) {
            $this->showAlreadyInstalled();
            return;
        }
        
        // Process form submissions
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->processStep();
        }
        
        // Display current step
        $this->displayStep();
    }
    
    private function processStep() {
        switch ($this->currentStep) {
            case 1:
                $this->processRequirements();
                break;
            case 2:
                $this->processDatabase();
                break;
            case 3:
                $this->processAdminSetup();
                break;
            case 4:
                $this->processSiteConfig();
                break;
            case 5:
                $this->processPaymentGateways();
                break;
        }
    }
    
    private function processRequirements() {
        $this->redirect(2);
    }
    
    private function processDatabase() {
        $host = $_POST['db_host'] ?? '';
        $name = $_POST['db_name'] ?? '';
        $user = $_POST['db_user'] ?? '';
        $pass = $_POST['db_pass'] ?? '';
        $prefix = $_POST['db_prefix'] ?? 'billing_';
        
        if (empty($host) || empty($name) || empty($user)) {
            $this->errors[] = 'Please fill in all required database fields.';
            return;
        }
        
        try {
            $dsn = "mysql:host={$host};dbname={$name};charset=utf8mb4";
            $pdo = new PDO($dsn, $user, $pass, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
            ]);
            
            $this->createDatabaseTables($pdo, $prefix);
            $this->saveDatabaseConfig($host, $name, $user, $pass, $prefix);
            
            $_SESSION['install_data']['database'] = [
                'host' => $host, 'name' => $name, 'user' => $user, 
                'pass' => $pass, 'prefix' => $prefix
            ];
            
            $this->success[] = 'Database configured successfully!';
            $this->redirect(3);
            
        } catch (Exception $e) {
            $this->errors[] = 'Database connection failed: ' . $e->getMessage();
        }
    }
    
    private function processAdminSetup() {
        $firstName = $_POST['admin_first_name'] ?? '';
        $lastName = $_POST['admin_last_name'] ?? '';
        $email = $_POST['admin_email'] ?? '';
        $password = $_POST['admin_password'] ?? '';
        $confirmPassword = $_POST['admin_password_confirm'] ?? '';
        
        if (empty($firstName) || empty($lastName) || empty($email) || empty($password)) {
            $this->errors[] = 'Please fill in all admin account fields.';
            return;
        }
        
        if ($password !== $confirmPassword) {
            $this->errors[] = 'Passwords do not match.';
            return;
        }
        
        if (strlen($password) < 8) {
            $this->errors[] = 'Password must be at least 8 characters long.';
            return;
        }
        
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->errors[] = 'Please enter a valid email address.';
            return;
        }
        
        try {
            $this->createAdminUser($firstName, $lastName, $email, $password);
            $_SESSION['install_data']['admin'] = [
                'first_name' => $firstName, 'last_name' => $lastName, 'email' => $email
            ];
            $this->success[] = 'Admin account created successfully!';
            $this->redirect(4);
        } catch (Exception $e) {
            $this->errors[] = 'Failed to create admin account: ' . $e->getMessage();
        }
    }
    
    private function processSiteConfig() {
        $siteName = $_POST['site_name'] ?? '';
        $siteUrl = $_POST['site_url'] ?? '';
        $companyName = $_POST['company_name'] ?? '';
        $timezone = $_POST['timezone'] ?? 'UTC';
        $currency = $_POST['currency'] ?? 'USD';
        
        if (empty($siteName) || empty($siteUrl) || empty($companyName)) {
            $this->errors[] = 'Please fill in all required site configuration fields.';
            return;
        }
        
        $_SESSION['install_data']['site'] = [
            'name' => $siteName, 'url' => rtrim($siteUrl, '/'),
            'company' => $companyName, 'timezone' => $timezone, 'currency' => $currency
        ];
        
        $this->success[] = 'Site configuration saved successfully!';
        $this->redirect(5);
    }
    
    private function processPaymentGateways() {
        $gateways = [
            'mpesa' => [
                'enabled' => isset($_POST['mpesa_enabled']),
                'consumer_key' => $_POST['mpesa_consumer_key'] ?? '',
                'consumer_secret' => $_POST['mpesa_consumer_secret'] ?? '',
                'shortcode' => $_POST['mpesa_shortcode'] ?? '',
                'passkey' => $_POST['mpesa_passkey'] ?? '',
                'mode' => $_POST['mpesa_mode'] ?? 'sandbox'
            ],
            'paypal' => [
                'enabled' => isset($_POST['paypal_enabled']),
                'client_id' => $_POST['paypal_client_id'] ?? '',
                'client_secret' => $_POST['paypal_client_secret'] ?? '',
                'mode' => $_POST['paypal_mode'] ?? 'sandbox'
            ],
            'stripe' => [
                'enabled' => isset($_POST['stripe_enabled']),
                'publishable_key' => $_POST['stripe_publishable_key'] ?? '',
                'secret_key' => $_POST['stripe_secret_key'] ?? ''
            ]
        ];
        
        $_SESSION['install_data']['gateways'] = $gateways;
        $this->completeInstallation();
        $this->redirect(6);
    }
    
    private function createDatabaseTables($pdo, $prefix) {
        $tables = [
            "CREATE TABLE IF NOT EXISTS {$prefix}users (
                id INT AUTO_INCREMENT PRIMARY KEY,
                first_name VARCHAR(100) NOT NULL,
                last_name VARCHAR(100) NOT NULL,
                email VARCHAR(255) UNIQUE NOT NULL,
                password VARCHAR(255) NOT NULL,
                role ENUM('admin', 'customer') DEFAULT 'customer',
                status ENUM('active', 'inactive', 'suspended') DEFAULT 'active',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            )",
            
            "CREATE TABLE IF NOT EXISTS {$prefix}customers (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT,
                company_name VARCHAR(255),
                phone VARCHAR(50),
                address TEXT,
                city VARCHAR(100),
                state VARCHAR(100),
                country VARCHAR(100),
                postal_code VARCHAR(20),
                tax_id VARCHAR(100),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES {$prefix}users(id) ON DELETE CASCADE
            )",
            
            "CREATE TABLE IF NOT EXISTS {$prefix}products (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(255) NOT NULL,
                description TEXT,
                type ENUM('domain', 'hosting', 'ssl', 'other') NOT NULL,
                price DECIMAL(10,2) NOT NULL,
                billing_cycle ENUM('monthly', 'quarterly', 'semi-annually', 'annually', 'biennially') NOT NULL,
                setup_fee DECIMAL(10,2) DEFAULT 0.00,
                status ENUM('active', 'inactive') DEFAULT 'active',
                auto_setup BOOLEAN DEFAULT TRUE,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )",
            
            "CREATE TABLE IF NOT EXISTS {$prefix}orders (
                id INT AUTO_INCREMENT PRIMARY KEY,
                customer_id INT NOT NULL,
                status ENUM('pending', 'active', 'suspended', 'cancelled') DEFAULT 'pending',
                total DECIMAL(10,2) NOT NULL,
                currency VARCHAR(3) DEFAULT 'USD',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (customer_id) REFERENCES {$prefix}customers(id)
            )",
            
            "CREATE TABLE IF NOT EXISTS {$prefix}invoices (
                id INT AUTO_INCREMENT PRIMARY KEY,
                customer_id INT NOT NULL,
                order_id INT,
                invoice_number VARCHAR(50) UNIQUE NOT NULL,
                subtotal DECIMAL(10,2) NOT NULL,
                tax_rate DECIMAL(5,2) DEFAULT 0.00,
                tax_amount DECIMAL(10,2) DEFAULT 0.00,
                total DECIMAL(10,2) NOT NULL,
                currency VARCHAR(3) DEFAULT 'USD',
                status ENUM('unpaid', 'paid', 'cancelled', 'refunded', 'pending_verification') DEFAULT 'unpaid',
                due_date DATE NOT NULL,
                paid_date DATETIME NULL,
                transaction_id VARCHAR(255),
                payment_method VARCHAR(50),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (customer_id) REFERENCES {$prefix}customers(id),
                FOREIGN KEY (order_id) REFERENCES {$prefix}orders(id)
            )",
            
            "CREATE TABLE IF NOT EXISTS {$prefix}invoice_items (
                id INT AUTO_INCREMENT PRIMARY KEY,
                invoice_id INT NOT NULL,
                type ENUM('domain', 'hosting', 'ssl', 'other') NOT NULL,
                description VARCHAR(255) NOT NULL,
                quantity INT DEFAULT 1,
                price DECIMAL(10,2) NOT NULL,
                domain VARCHAR(255),
                period INT,
                action ENUM('register', 'renew', 'transfer') DEFAULT 'register',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (invoice_id) REFERENCES {$prefix}invoices(id) ON DELETE CASCADE
            )",
            
            "CREATE TABLE IF NOT EXISTS {$prefix}services (
                id INT AUTO_INCREMENT PRIMARY KEY,
                customer_id INT NOT NULL,
                order_id INT,
                product_id INT NOT NULL,
                domain VARCHAR(255),
                username VARCHAR(100),
                password VARCHAR(255),
                status ENUM('pending', 'active', 'suspended', 'cancelled') DEFAULT 'pending',
                next_due_date DATE,
                billing_cycle ENUM('monthly', 'quarterly', 'semi-annually', 'annually', 'biennially') NOT NULL,
                price DECIMAL(10,2) NOT NULL,
                details JSON,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (customer_id) REFERENCES {$prefix}customers(id),
                FOREIGN KEY (order_id) REFERENCES {$prefix}orders(id),
                FOREIGN KEY (product_id) REFERENCES {$prefix}products(id)
            )",
            
            "CREATE TABLE IF NOT EXISTS {$prefix}transactions (
                id INT AUTO_INCREMENT PRIMARY KEY,
                invoice_id INT,
                customer_id INT NOT NULL,
                transaction_id VARCHAR(255) UNIQUE NOT NULL,
                gateway VARCHAR(50) NOT NULL,
                amount DECIMAL(10,2) NOT NULL,
                currency VARCHAR(3) DEFAULT 'USD',
                status ENUM('pending', 'completed', 'failed', 'cancelled', 'refunded') DEFAULT 'pending',
                gateway_response TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (invoice_id) REFERENCES {$prefix}invoices(id),
                FOREIGN KEY (customer_id) REFERENCES {$prefix}customers(id)
            )",
            
            "CREATE TABLE IF NOT EXISTS {$prefix}mpesa_transactions (
                id INT AUTO_INCREMENT PRIMARY KEY,
                checkout_request_id VARCHAR(255) UNIQUE NOT NULL,
                merchant_request_id VARCHAR(255),
                invoice_id INT,
                phone_number VARCHAR(20),
                amount DECIMAL(10,2) NOT NULL,
                status ENUM('pending', 'success', 'failed') DEFAULT 'pending',
                transaction_id VARCHAR(255),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (invoice_id) REFERENCES {$prefix}invoices(id)
            )",
            
            "CREATE TABLE IF NOT EXISTS {$prefix}settings (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(255) UNIQUE NOT NULL,
                value TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            )"
        ];
        
        foreach ($tables as $sql) {
            $pdo->exec($sql);
        }
    }
    
    private function createAdminUser($firstName, $lastName, $email, $password) {
        $config = include 'config/database.php';
        $pdo = new PDO($config['dsn'], $config['username'], $config['password'], $config['options']);
        
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("INSERT INTO {$config['prefix']}users (first_name, last_name, email, password, role) VALUES (?, ?, ?, ?, 'admin')");
        $stmt->execute([$firstName, $lastName, $email, $hashedPassword]);
    }
    
    private function saveDatabaseConfig($host, $name, $user, $pass, $prefix) {
        $config = "<?php
\$dsn = 'mysql:host={$host};dbname={$name};charset=utf8mb4';
\$username = '{$user}';
\$password = '{$pass}';
\$prefix = '{$prefix}';

\$options = [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    PDO::ATTR_EMULATE_PREPARES => false,
];

return [
    'dsn' => \$dsn,
    'username' => \$username,
    'password' => \$password,
    'prefix' => \$prefix,
    'options' => \$options
];
?>";
        
        if (!is_dir('config')) {
            mkdir('config', 0755, true);
        }
        file_put_contents('config/database.php', $config);
    }
    
    private function completeInstallation() {
        $this->createMainConfig();
        $this->createHtaccess();
        file_put_contents('config/installed.lock', date('Y-m-d H:i:s'));
        unset($_SESSION['install_data']);
    }
    
    private function createMainConfig() {
        $installData = $_SESSION['install_data'];
        $config = "<?php
return [
    'site_url' => '{$installData['site']['url']}',
    'site_name' => '{$installData['site']['name']}',
    'company_name' => '{$installData['site']['company']}',
    'timezone' => '{$installData['site']['timezone']}',
    'currency' => '{$installData['site']['currency']}',
    
    'mpesa' => [
        'enabled' => " . ($installData['gateways']['mpesa']['enabled'] ? 'true' : 'false') . ",
        'mode' => '{$installData['gateways']['mpesa']['mode']}',
        'consumer_key' => '{$installData['gateways']['mpesa']['consumer_key']}',
        'consumer_secret' => '{$installData['gateways']['mpesa']['consumer_secret']}',
        'shortcode' => '{$installData['gateways']['mpesa']['shortcode']}',
        'passkey' => '{$installData['gateways']['mpesa']['passkey']}',
        'callback_url' => '{$installData['site']['url']}/payment_process.php?action=mpesa_callback'
    ],
    
    'paypal' => [
        'enabled' => " . ($installData['gateways']['paypal']['enabled'] ? 'true' : 'false') . ",
        'mode' => '{$installData['gateways']['paypal']['mode']}',
        'client_id' => '{$installData['gateways']['paypal']['client_id']}',
        'client_secret' => '{$installData['gateways']['paypal']['client_secret']}'
    ],
    
    'stripe' => [
        'enabled' => " . ($installData['gateways']['stripe']['enabled'] ? 'true' : 'false') . ",
        'publishable_key' => '{$installData['gateways']['stripe']['publishable_key']}',
        'secret_key' => '{$installData['gateways']['stripe']['secret_key']}'
    ],
    
    'domain_apis' => [
        'default_registrar' => 'namecheap',
        'namecheap' => ['mode' => 'sandbox', 'api_user' => '', 'api_key' => '', 'username' => '']
    ],
    
    'hosting_apis' => [
        'default_provider' => 'cpanel',
        'cpanel' => ['server' => '', 'server_ip' => '', 'username' => '', 'api_token' => '', 'nameservers' => []]
    ],
    
    'email' => [
        'smtp_host' => 'localhost',
        'smtp_port' => 587,
        'smtp_username' => '',
        'smtp_password' => '',
        'smtp_encryption' => 'tls',
        'from_email' => 'noreply@' . parse_url('{$installData['site']['url']}', PHP_URL_HOST),
        'from_name' => '{$installData['site']['name']}'
    ],
    
    'security' => [
        'encryption_key' => '" . bin2hex(random_bytes(16)) . "',
        'hash_algorithm' => 'sha256',
        'session_timeout' => 3600
    ]
];
?>";
        file_put_contents('config/payment_config.php', $config);
    }
    
    private function createHtaccess() {
        $htaccess = "RewriteEngine On
<Files ~ \"^\\.(htaccess|htpasswd)$\">
    deny from all
</Files>
<Directory \"config\">
    deny from all
</Directory>
<Directory \"includes\">
    deny from all
</Directory>
<Directory \"classes\">
    deny from all
</Directory>";
        file_put_contents('.htaccess', $htaccess);
    }
    
    private function checkRequirements() {
        return [
            'PHP Version >= 7.4' => version_compare(PHP_VERSION, '7.4.0', '>='),
            'PDO Extension' => extension_loaded('pdo'),
            'PDO MySQL Extension' => extension_loaded('pdo_mysql'),
            'cURL Extension' => extension_loaded('curl'),
            'OpenSSL Extension' => extension_loaded('openssl'),
            'JSON Extension' => extension_loaded('json'),
            'Config Directory Writable' => is_writable('.') || is_writable('config'),
            'Sessions Enabled' => function_exists('session_start')
        ];
    }
    
    private function displayStep() {
        $requirements = $this->checkRequirements();
        $allRequirementsMet = !in_array(false, $requirements);
        ?>
        <!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>Billing System Installation</title>
            <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
        </head>
        <body class="bg-light">
            <div class="container">
                <div class="row justify-content-center">
                    <div class="col-md-8">
                        <div class="card mt-5">
                            <div class="card-header">
                                <h3>🚀 Billing System Installation</h3>
                                <div class="progress mt-3">
                                    <div class="progress-bar" style="width: <?= ($this->currentStep / count($this->steps)) * 100 ?>%"></div>
                                </div>
                                <p class="mt-2 mb-0"><strong>Step <?= $this->currentStep ?> of <?= count($this->steps) ?>:</strong> <?= $this->steps[$this->currentStep] ?></p>
                            </div>
                            <div class="card-body">
                                <?php if (!empty($this->errors)): ?>
                                    <div class="alert alert-danger">
                                        <ul class="mb-0">
                                            <?php foreach ($this->errors as $error): ?>
                                                <li><?= htmlspecialchars($error) ?></li>
                                            <?php endforeach; ?>
                                        </ul>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($this->success)): ?>
                                    <div class="alert alert-success">
                                        <?php foreach ($this->success as $msg): ?>
                                            <p class="mb-0"><?= htmlspecialchars($msg) ?></p>
                                        <?php endforeach; ?>
                                    </div>
                                <?php endif; ?>
                                
                                <?php
                                switch ($this->currentStep) {
                                    case 1: $this->displayRequirements($requirements, $allRequirementsMet); break;
                                    case 2: $this->displayDatabaseConfig(); break;
                                    case 3: $this->displayAdminSetup(); break;
                                    case 4: $this->displaySiteConfig(); break;
                                    case 5: $this->displayPaymentGateways(); break;
                                    case 6: $this->displayComplete(); break;
                                }
                                ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </body>
        </html>
        <?php
    }
    
    private function displayRequirements($requirements, $allMet) {
        ?>
        <h4>System Requirements Check</h4>
        <ul class="list-group">
            <?php foreach ($requirements as $req => $status): ?>
                <li class="list-group-item d-flex justify-content-between align-items-center">
                    <?= htmlspecialchars($req) ?>
                    <span class="badge bg-<?= $status ? 'success' : 'danger' ?>"><?= $status ? '✓' : '✗' ?></span>
                </li>
            <?php endforeach; ?>
        </ul>
        
        <div class="d-flex justify-content-end mt-3">
            <?php if ($allMet): ?>
                <form method="post">
                    <button type="submit" class="btn btn-primary">Continue →</button>
                </form>
            <?php else: ?>
                <p class="text-danger">Please resolve failed requirements before continuing.</p>
            <?php endif; ?>
        </div>
        <?php
    }
    
    private function displayDatabaseConfig() {
        ?>
        <h4>Database Configuration</h4>
        <form method="post">
            <div class="row">
                <div class="col-md-6 mb-3">
                    <label class="form-label">Database Host</label>
                    <input type="text" name="db_host" class="form-control" value="localhost" required>
                </div>
                <div class="col-md-6 mb-3">
                    <label class="form-label">Database Name</label>
                    <input type="text" name="db_name" class="form-control" required>
                </div>
            </div>
            <div class="row">
                <div class="col-md-6 mb-3">
                    <label class="form-label">Database Username</label>
                    <input type="text" name="db_user" class="form-control" required>
                </div>
                <div class="col-md-6 mb-3">
                    <label class="form-label">Database Password</label>
                    <input type="password" name="db_pass" class="form-control">
                </div>
            </div>
            <div class="mb-3">
                <label class="form-label">Table Prefix</label>
                <input type="text" name="db_prefix" class="form-control" value="billing_">
            </div>
            <div class="d-flex justify-content-between">
                <a href="?step=1" class="btn btn-secondary">← Back</a>
                <button type="submit" class="btn btn-primary">Continue →</button>
            </div>
        </form>
        <?php
    }
    
    private function displayAdminSetup() {
        ?>
        <h4>Administrator Account</h4>
        <form method="post">
            <div class="row">
                <div class="col-md-6 mb-3">
                    <label class="form-label">First Name</label>
                    <input type="text" name="admin_first_name" class="form-control" required>
                </div>
                <div class="col-md-6 mb-3">
                    <label class="form-label">Last Name</label>
                    <input type="text" name="admin_last_name" class="form-control" required>
                </div>
            </div>
            <div class="mb-3">
                <label class="form-label">Email Address</label>
                <input type="email" name="admin_email" class="form-control" required>
            </div>
            <div class="row">
                <div class="col-md-6 mb-3">
                    <label class="form-label">Password</label>
                    <input type="password" name="admin_password" class="form-control" required minlength="8">
                </div>
                <div class="col-md-6 mb-3">
                    <label class="form-label">Confirm Password</label>
                    <input type="password" name="admin_password_confirm" class="form-control" required>
                </div>
            </div>
            <div class="d-flex justify-content-between">
                <a href="?step=2" class="btn btn-secondary">← Back</a>
                <button type="submit" class="btn btn-primary">Continue →</button>
            </div>
        </form>
        <?php
    }
    
    private function displaySiteConfig() {
        ?>
        <h4>Site Configuration</h4>
        <form method="post">
            <div class="mb-3">
                <label class="form-label">Site Name</label>
                <input type="text" name="site_name" class="form-control" required>
            </div>
            <div class="mb-3">
                <label class="form-label">Site URL</label>
                <input type="url" name="site_url" class="form-control" required value="<?= 'http' . (isset($_SERVER['HTTPS']) ? 's' : '') . '://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['REQUEST_URI']) ?>">
            </div>
            <div class="mb-3">
                <label class="form-label">Company Name</label>
                <input type="text" name="company_name" class="form-control" required>
            </div>
            <div class="row">
                <div class="col-md-6 mb-3">
                    <label class="form-label">Timezone</label>
                    <select name="timezone" class="form-control">
                        <option value="UTC">UTC</option>
                        <option value="Africa/Nairobi" selected>Nairobi</option>
                        <option value="America/New_York">New York</option>
                        <option value="Europe/London">,
        'EUR' => '€', 
        'GBP' => '£',
        'KES' => 'KSh ',
        'UGX' => 'USh ',
        'TZS' => 'TSh '
    ];
    
    $symbol = $symbols[$currency] ?? $currency . ' ';
    return $symbol . number_format($amount, 2);
}

function sendEmail($to, $subject, $message, $headers = []) {
    $config = include __DIR__ . '/../config/payment_config.php';
    
    $defaultHeaders = [
        'From' => $config['email']['from_name'] . ' <' . $config['email']['from_email'] . '>',
        'Reply-To' => $config['email']['from_email'],
        'X-Mailer' => 'Billing System',
        'MIME-Version' => '1.0',
        'Content-Type' => 'text/html; charset=UTF-8'
    ];
    
    $allHeaders = array_merge($defaultHeaders, $headers);
    $headerString = '';
    foreach ($allHeaders as $key => $value) {
        $headerString .= $key . ': ' . $value . "\r\n";
    }
    
    return mail($to, $subject, $message, $headerString);
}

function logActivity($message, $level = 'info') {
    $logFile = __DIR__ . '/../logs/system.log';
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[{$timestamp}] [{$level}] {$message}" . PHP_EOL;
    
    if (!is_dir(dirname($logFile))) {
        mkdir(dirname($logFile), 0755, true);
    }
    
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

function requireLogin() {
    if (!isLoggedIn()) {
        header('Location: /login.php');
        exit;
    }
}

function isAdmin() {
    return isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin';
}

function requireAdmin() {
    requireLogin();
    if (!isAdmin()) {
        header('Location: /customer/dashboard.php');
        exit;
    }
}

function sanitizeInput($input) {
    if (is_array($input)) {
        return array_map('sanitizeInput', $input);
    }
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

function getDatabase() {
    static $pdo = null;
    
    if ($pdo === null) {
        $config = include __DIR__ . '/../config/database.php';
        
        try {
            $pdo = new PDO($config['dsn'], $config['username'], $config['password'], $config['options']);
        } catch (PDOException $e) {
            error_log('Database connection failed: ' . $e->getMessage());
            die('Database connection failed. Please check your configuration.');
        }
    }
    
    return $pdo;
}
?>