<?php
/**
 * Simple Admin Dashboard
 */

session_start();

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'admin') {
    header('Location: ../login.php?return=' . urlencode($_SERVER['REQUEST_URI']));
    exit;
}

$userName = $_SESSION['user_name'] ?? 'Admin';
$userEmail = $_SESSION['user_email'] ?? '';

// Get basic stats
$stats = [
    'total_users' => 0,
    'total_invoices' => 0,
    'total_revenue' => 0,
    'pending_payments' => 0
];

try {
    $config = include '../config/database.php';
    $pdo = new PDO($config['dsn'], $config['username'], $config['password'], $config['options']);
    
    // Count total users
    $stmt = $pdo->query("SELECT COUNT(*) FROM {$config['prefix']}users WHERE role = 'customer'");
    $stats['total_users'] = $stmt->fetchColumn();
    
    // Count total invoices
    $stmt = $pdo->query("SELECT COUNT(*) FROM {$config['prefix']}invoices");
    $stats['total_invoices'] = $stmt->fetchColumn();
    
    // Calculate total revenue
    $stmt = $pdo->query("SELECT SUM(total) FROM {$config['prefix']}invoices WHERE status = 'paid'");
    $stats['total_revenue'] = $stmt->fetchColumn() ?: 0;
    
    // Count pending payments
    $stmt = $pdo->query("SELECT COUNT(*) FROM {$config['prefix']}invoices WHERE status = 'unpaid'");
    $stats['pending_payments'] = $stmt->fetchColumn();
    
} catch (Exception $e) {
    error_log("Admin dashboard error: " . $e->getMessage());
}

function formatCurrency($amount, $currency = 'USD') {
    $symbols = ['USD' => '$', 'EUR' => '€', 'GBP' => '£', 'KES' => 'KSh '];
    $symbol = $symbols[$currency] ?? $currency . ' ';
    return $symbol . number_format($amount, 2);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Billing System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    
    <style>
        :root {
            --primary-gradient: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            --sidebar-bg: #2c3e50;
            --sidebar-hover: #34495e;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #f8f9fa;
        }
        
        .sidebar {
            background: var(--sidebar-bg);
            min-height: 100vh;
            width: 250px;
            position: fixed;
            top: 0;
            left: 0;
            z-index: 1000;
            transition: all 0.3s ease;
        }
        
        .sidebar .sidebar-header {
            padding: 20px;
            border-bottom: 1px solid #34495e;
        }
        
        .sidebar .sidebar-header h4 {
            color: white;
            margin: 0;
            font-weight: 600;
        }
        
        .sidebar-nav {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        
        .sidebar-nav li {
            border-bottom: 1px solid #34495e;
        }
        
        .sidebar-nav a {
            display: block;
            padding: 15px 20px;
            color: #bdc3c7;
            text-decoration: none;
            transition: all 0.3s ease;
        }
        
        .sidebar-nav a:hover,
        .sidebar-nav a.active {
            background: var(--sidebar-hover);
            color: white;
            padding-left: 30px;
        }
        
        .sidebar-nav a i {
            width: 20px;
            margin-right: 10px;
        }
        
        .main-content {
            margin-left: 250px;
            padding: 0;
            min-height: 100vh;
        }
        
        .top-navbar {
            background: white;
            padding: 15px 30px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            border-left: 4px solid #667eea;
        }
        
        .dashboard-content {
            padding: 30px;
        }
        
        .stat-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            border-left: 4px solid;
            transition: transform 0.3s ease;
            height: 100%;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
        }
        
        .stat-card.primary { border-left-color: #667eea; }
        .stat-card.success { border-left-color: #28a745; }
        .stat-card.warning { border-left-color: #ffc107; }
        .stat-card.danger { border-left-color: #dc3545; }
        
        .stat-card .stat-icon {
            font-size: 2.5rem;
            opacity: 0.7;
        }
        
        .stat-card .stat-number {
            font-size: 2rem;
            font-weight: 700;
            margin: 10px 0;
        }
        
        .stat-card .stat-label {
            color: #6c757d;
            font-size: 0.9rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .quick-actions {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .quick-actions .btn {
            margin: 5px;
            border-radius: 25px;
            padding: 10px 20px;
            font-weight: 500;
        }
        
        .recent-activity {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .activity-item {
            padding: 15px 0;
            border-bottom: 1px solid #eee;
        }
        
        .activity-item:last-child {
            border-bottom: none;
        }
        
        .user-dropdown .dropdown-toggle::after {
            display: none;
        }
        
        @media (max-width: 768px) {
            .sidebar {
                margin-left: -250px;
            }
            
            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <h4><i class="fas fa-credit-card me-2"></i>Admin Panel</h4>
        </div>
        
        <ul class="sidebar-nav">
            <li><a href="index.php" class="active"><i class="fas fa-tachometer-alt"></i>Dashboard</a></li>
            <li><a href="customers.php"><i class="fas fa-users"></i>Customers</a></li>
            <li><a href="invoices.php"><i class="fas fa-file-invoice"></i>Invoices</a></li>
            <li><a href="payments.php"><i class="fas fa-credit-card"></i>Payments</a></li>
            <li><a href="products.php"><i class="fas fa-box"></i>Products</a></li>
            <li><a href="reports.php"><i class="fas fa-chart-bar"></i>Reports</a></li>
            <li><a href="settings.php"><i class="fas fa-cog"></i>Settings</a></li>
            <li><a href="help.php"><i class="fas fa-question-circle"></i>Help</a></li>
        </ul>
    </div>
    
    <!-- Main Content -->
    <div class="main-content">
        <!-- Top Navigation -->
        <div class="top-navbar d-flex justify-content-between align-items-center">
            <div>
                <h3 class="mb-0">Dashboard Overview</h3>
                <small class="text-muted">Welcome back, <?= htmlspecialchars($userName) ?>!</small>
            </div>
            
            <div class="d-flex align-items-center">
                <div class="dropdown">
                    <button class="btn btn-outline-primary dropdown-toggle user-dropdown" type="button" data-bs-toggle="dropdown">
                        <i class="fas fa-user-circle me-2"></i><?= htmlspecialchars(explode(' ', $userName)[0]) ?>
                    </button>
                    <ul class="dropdown-menu">
                        <li><a class="dropdown-item" href="profile.php"><i class="fas fa-user me-2"></i>Profile</a></li>
                        <li><a class="dropdown-item" href="settings.php"><i class="fas fa-cog me-2"></i>Settings</a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="../logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
                    </ul>
                </div>
            </div>
        </div>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <!-- Statistics Cards -->
            <div class="row mb-4">
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="stat-card primary">
                        <div class="d-flex justify-content-between">
                            <div>
                                <div class="stat-label">Total Customers</div>
                                <div class="stat-number text-primary"><?= number_format($stats['total_users']) ?></div>
                            </div>
                            <div class="stat-icon text-primary">
                                <i class="fas fa-users"></i>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="stat-card success">
                        <div class="d-flex justify-content-between">
                            <div>
                                <div class="stat-label">Total Revenue</div>
                                <div class="stat-number text-success"><?= formatCurrency($stats['total_revenue']) ?></div>
                            </div>
                            <div class="stat-icon text-success">
                                <i class="fas fa-dollar-sign"></i>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="stat-card warning">
                        <div class="d-flex justify-content-between">
                            <div>
                                <div class="stat-label">Pending Payments</div>
                                <div class="stat-number text-warning"><?= number_format($stats['pending_payments']) ?></div>
                            </div>
                            <div class="stat-icon text-warning">
                                <i class="fas fa-clock"></i>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="stat-card danger">
                        <div class="d-flex justify-content-between">
                            <div>
                                <div class="stat-label">Total Invoices</div>
                                <div class="stat-number text-danger"><?= number_format($stats['total_invoices']) ?></div>
                            </div>
                            <div class="stat-icon text-danger">
                                <i class="fas fa-file-invoice"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Quick Actions and Recent Activity -->
            <div class="row">
                <div class="col-lg-6 mb-4">
                    <div class="quick-actions">
                        <h5 class="mb-4"><i class="fas fa-rocket me-2"></i>Quick Actions</h5>
                        
                        <div class="d-grid gap-2">
                            <a href="customers.php?action=add" class="btn btn-primary">
                                <i class="fas fa-user-plus me-2"></i>Add New Customer
                            </a>
                            <a href="invoices.php?action=create" class="btn btn-success">
                                <i class="fas fa-file-plus me-2"></i>Create Invoice
                            </a>
                            <a href="products.php?action=add" class="btn btn-info">
                                <i class="fas fa-box me-2"></i>Add Product
                            </a>
                            <a href="payments.php" class="btn btn-warning">
                                <i class="fas fa-search me-2"></i>View Payments
                            </a>
                            <a href="reports.php" class="btn btn-secondary">
                                <i class="fas fa-chart-line me-2"></i>Generate Report
                            </a>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-6 mb-4">
                    <div class="recent-activity">
                        <h5 class="mb-4"><i class="fas fa-clock me-2"></i>Recent Activity</h5>
                        
                        <div class="activity-item">
                            <div class="d-flex align-items-center">
                                <div class="me-3">
                                    <i class="fas fa-user-check text-success"></i>
                                </div>
                                <div>
                                    <strong>System Installed</strong><br>
                                    <small class="text-muted">Admin account created successfully</small>
                                </div>
                            </div>
                        </div>
                        
                        <div class="activity-item">
                            <div class="d-flex align-items-center">
                                <div class="me-3">
                                    <i class="fas fa-database text-primary"></i>
                                </div>
                                <div>
                                    <strong>Database Connected</strong><br>
                                    <small class="text-muted">All tables created and ready</small>
                                </div>
                            </div>
                        </div>
                        
                        <div class="activity-item">
                            <div class="d-flex align-items-center">
                                <div class="me-3">
                                    <i class="fas fa-shield-alt text-info"></i>
                                </div>
                                <div>
                                    <strong>Security Enabled</strong><br>
                                    <small class="text-muted">System security features activated</small>
                                </div>
                            </div>
                        </div>
                        
                        <div class="text-center mt-3">
                            <a href="activity.php" class="btn btn-sm btn-outline-primary">View All Activity</a>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- System Status -->
            <div class="row">
                <div class="col-12">
                    <div class="quick-actions">
                        <h5 class="mb-4"><i class="fas fa-heartbeat me-2"></i>System Status</h5>
                        
                        <div class="row">
                            <div class="col-md-3">
                                <div class="text-center">
                                    <i class="fas fa-database fa-2x text-success mb-2"></i>
                                    <h6>Database</h6>
                                    <span class="badge bg-success">Connected</span>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="text-center">
                                    <i class="fas fa-credit-card fa-2x text-warning mb-2"></i>
                                    <h6>Payment Gateways</h6>
                                    <span class="badge bg-warning">Setup Required</span>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="text-center">
                                    <i class="fas fa-envelope fa-2x text-warning mb-2"></i>
                                    <h6>Email System</h6>
                                    <span class="badge bg-warning">Setup Required</span>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="text-center">
                                    <i class="fas fa-shield-alt fa-2x text-success mb-2"></i>
                                    <h6>Security</h6>
                                    <span class="badge bg-success">Active</span>
                                </div>
                            </div>
                        </div>
                        
                        <div class="text-center mt-4">
                            <a href="settings.php" class="btn btn-primary">
                                <i class="fas fa-cog me-2"></i>Configure System Settings
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        // Auto-refresh stats every 30 seconds
        setInterval(() => {
            // You can add AJAX calls here to refresh statistics
            console.log('Auto-refresh triggered');
        }, 30000);
        
        // Add some interactive features
        document.querySelectorAll('.stat-card').forEach(card => {
            card.addEventListener('click', function() {
                // Add click handlers for stat cards
                console.log('Stat card clicked:', this);
            });
        });
    </script>
</body>
</html>