<?php
/**
 * Debug Installation Script
 * Upload this file alongside your billing system and run it to diagnose issues
 */

echo "<h1>🔍 Billing System Installation Diagnostics</h1>";
echo "<style>body{font-family:Arial;margin:20px;} .pass{color:green;} .fail{color:red;} .warn{color:orange;}</style>";

// Check PHP version
echo "<h2>📋 System Requirements</h2>";
echo "<table border='1' cellpadding='5'>";
echo "<tr><th>Requirement</th><th>Status</th><th>Details</th></tr>";

// PHP Version
$phpVersion = PHP_VERSION;
$phpOk = version_compare($phpVersion, '7.4.0', '>=');
echo "<tr><td>PHP Version >= 7.4</td>";
echo "<td class='" . ($phpOk ? 'pass'>✅ PASS' : 'fail'>❌ FAIL') . "</td>";
echo "<td>Current: {$phpVersion}</td></tr>";

// Extensions
$extensions = ['pdo', 'pdo_mysql', 'curl', 'openssl', 'json'];
foreach ($extensions as $ext) {
    $loaded = extension_loaded($ext);
    echo "<tr><td>{$ext} Extension</td>";
    echo "<td class='" . ($loaded ? 'pass'>✅ PASS' : 'fail'>❌ FAIL') . "</td>";
    echo "<td>" . ($loaded ? 'Loaded' : 'Missing') . "</td></tr>";
}

// Directory permissions
$dirs = ['.', 'config', 'includes', 'classes'];
foreach ($dirs as $dir) {
    $exists = is_dir($dir);
    $writable = is_writable($dir);
    echo "<tr><td>{$dir} Directory</td>";
    echo "<td class='" . ($exists && $writable ? 'pass'>✅ PASS' : ($exists ? 'warn'>⚠️ WARN' : 'fail'>❌ FAIL')) . "</td>";
    echo "<td>";
    if (!$exists) echo "Does not exist";
    elseif (!$writable) echo "Not writable (set to 755)";
    else echo "OK";
    echo "</td></tr>";
}

echo "</table>";

// Check critical files
echo "<h2>📁 File Structure Check</h2>";
$criticalFiles = [
    'install.php',
    'index.php',
    'payment_process.php',
    'includes/functions.php',
    'includes/database.php',
    'classes/PaymentGateway.php'
];

echo "<table border='1' cellpadding='5'>";
echo "<tr><th>File</th><th>Status</th><th>Size</th></tr>";

foreach ($criticalFiles as $file) {
    $exists = file_exists($file);
    $size = $exists ? filesize($file) : 0;
    echo "<tr><td>{$file}</td>";
    echo "<td class='" . ($exists && $size > 0 ? 'pass'>✅ EXISTS' : 'fail'>❌ MISSING') . "</td>";
    echo "<td>" . ($exists ? number_format($size) . ' bytes' : 'N/A') . "</td></tr>";
}

echo "</table>";

// Database connection test
echo "<h2>🗄️ Database Connection Test</h2>";

if (file_exists('config/database.php')) {
    echo "<p class='warn'>⚠️ Database config already exists. Installation may be complete.</p>";
    
    try {
        $config = include 'config/database.php';
        $pdo = new PDO($config['dsn'], $config['username'], $config['password'], $config['options']);
        echo "<p class='pass'>✅ Database connection successful!</p>";
        
        // Check if tables exist
        $stmt = $pdo->query("SHOW TABLES LIKE '{$config['prefix']}users'");
        if ($stmt->rowCount() > 0) {
            echo "<p class='pass'>✅ Database tables exist.</p>";
            echo "<p><strong>Installation appears to be complete!</strong></p>";
            echo "<p>Try accessing: <a href='admin/'>Admin Panel</a> | <a href='login.php'>Login Page</a></p>";
        } else {
            echo "<p class='warn'>⚠️ Database connected but tables missing.</p>";
        }
        
    } catch (Exception $e) {
        echo "<p class='fail'>❌ Database connection failed: " . $e->getMessage() . "</p>";
    }
} else {
    echo "<p>ℹ️ No database config found. This is normal for first installation.</p>";
    echo "<p><strong>Next step:</strong> Create MySQL database in cPanel, then run installer.</p>";
}

// Installation status
echo "<h2>⚙️ Installation Status</h2>";

if (file_exists('config/installed.lock')) {
    $installDate = file_get_contents('config/installed.lock');
    echo "<p class='pass'>✅ Installation completed on: {$installDate}</p>";
    echo "<p><strong>System is ready to use!</strong></p>";
    echo "<p>Delete install.php for security.</p>";
} else {
    echo "<p class='warn'>⚠️ Installation not completed yet.</p>";
    echo "<p><strong>Next step:</strong> <a href='install.php'>Run the installer</a></p>";
}

// Environment info
echo "<h2>🖥️ Server Environment</h2>";
echo "<table border='1' cellpadding='5'>";
echo "<tr><th>Setting</th><th>Value</th></tr>";
echo "<tr><td>Server Software</td><td>" . ($_SERVER['SERVER_SOFTWARE'] ?? 'Unknown') . "</td></tr>";
echo "<tr><td>Document Root</td><td>" . ($_SERVER['DOCUMENT_ROOT'] ?? 'Unknown') . "</td></tr>";
echo "<tr><td>Current Directory</td><td>" . getcwd() . "</td></tr>";
echo "<tr><td>PHP Memory Limit</td><td>" . ini_get('memory_limit') . "</td></tr>";
echo "<tr><td>Max Execution Time</td><td>" . ini_get('max_execution_time') . "s</td></tr>";
echo "<tr><td>Upload Max Size</td><td>" . ini_get('upload_max_filesize') . "</td></tr>";
echo "</table>";

// Common issues and solutions
echo "<h2>🔧 Common Issues & Solutions</h2>";
echo "<div style='background: #f5f5f5; padding: 15px; border-radius: 5px;'>";
echo "<h3>Issue: Database connection failed</h3>";
echo "<p><strong>Solution:</strong></p>";
echo "<ol>";
echo "<li>Create database in cPanel → MySQL Databases</li>";
echo "<li>Use FULL database name: <code>cpanelusername_dbname</code></li>";
echo "<li>Create database user with ALL PRIVILEGES</li>";
echo "<li>Use <code>localhost</code> as host</li>";
echo "</ol>";

echo "<h3>Issue: Config directory not writable</h3>";
echo "<p><strong>Solution:</strong></p>";
echo "<ol>";
echo "<li>In cPanel File Manager, right-click 'config' folder</li>";
echo "<li>Select 'Change Permissions'</li>";
echo "<li>Set to 755 (rwxr-xr-x)</li>";
echo "</ol>";

echo "<h3>Issue: Files missing or corrupted</h3>";
echo "<p><strong>Solution:</strong></p>";
echo "<ol>";
echo "<li>Re-upload all files from ZIP</li>";
echo "<li>Ensure directory structure is correct</li>";
echo "<li>Check file sizes are not 0 bytes</li>";
echo "</ol>";

echo "<h3>Issue: Installation already exists</h3>";
echo "<p><strong>Solution:</strong></p>";
echo "<ol>";
echo "<li>Delete <code>config/installed.lock</code> file</li>";
echo "<li>Delete <code>config/database.php</code> if you want fresh install</li>";
echo "<li>Re-run installer</li>";
echo "</ol>";

echo "</div>";

// Quick actions
echo "<h2>🚀 Quick Actions</h2>";
echo "<p>";
if (!file_exists('config/installed.lock')) {
    echo "<a href='install.php' style='background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>🔧 Run Installer</a> ";
}
echo "<a href='index.php' style='background: #28a745; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>🏠 View Homepage</a> ";
if (file_exists('config/installed.lock')) {
    echo "<a href='admin/' style='background: #6f42c1; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>👨‍💼 Admin Panel</a> ";
}
echo "</p>";

echo "<hr>";
echo "<p><small>Debug script completed. Delete this file after troubleshooting.</small></p>";
?>