<?php
/**
 * Fixed Billing System Installer
 * Solves the redirect loop issue
 */

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Check if already installed
if (file_exists('config/installed.lock')) {
    echo '<!DOCTYPE html><html><head><title>Already Installed</title></head><body>';
    echo '<div style="text-align:center;padding:50px;font-family:Arial;">';
    echo '<h1>⚠️ Already Installed</h1>';
    echo '<p>The system is already installed!</p>';
    echo '<p><a href="test_login.php" style="background:#007bff;color:white;padding:10px 20px;text-decoration:none;border-radius:4px;">Test Login</a></p>';
    echo '<p><a href="index.php" style="background:#28a745;color:white;padding:10px 20px;text-decoration:none;border-radius:4px;">View Site</a></p>';
    echo '</div></body></html>';
    exit;
}

$currentStep = 1;
$error = '';
$success = '';

// Determine current step based on what's completed
if (file_exists('config/database.php')) {
    $currentStep = 2; // Database is set up, need admin user
    
    // Check if admin user exists
    try {
        $config = include 'config/database.php';
        $pdo = new PDO($config['dsn'], $config['username'], $config['password'], $config['options']);
        $stmt = $pdo->query("SELECT COUNT(*) as count FROM {$config['prefix']}users WHERE role = 'admin'");
        $result = $stmt->fetch();
        
        if ($result['count'] > 0) {
            $currentStep = 3; // Admin exists, installation complete
        }
    } catch (Exception $e) {
        // Database config exists but connection fails, stay at step 2
        $error = "Database connection issue: " . $e->getMessage();
        $currentStep = 1;
    }
}

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['setup_database'])) {
        // Step 1: Database setup
        $host = trim($_POST['db_host'] ?? '');
        $name = trim($_POST['db_name'] ?? '');
        $user = trim($_POST['db_user'] ?? '');
        $pass = $_POST['db_pass'] ?? '';
        $prefix = trim($_POST['db_prefix'] ?? 'billing_');
        
        if (empty($host) || empty($name) || empty($user)) {
            $error = 'Please fill in all required database fields.';
        } else {
            try {
                // Test database connection
                $dsn = "mysql:host={$host};dbname={$name};charset=utf8mb4";
                $pdo = new PDO($dsn, $user, $pass, [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
                ]);
                
                // Create tables
                $tables = [
                    "CREATE TABLE IF NOT EXISTS {$prefix}users (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        first_name VARCHAR(100) NOT NULL,
                        last_name VARCHAR(100) NOT NULL,
                        email VARCHAR(255) UNIQUE NOT NULL,
                        password VARCHAR(255) NOT NULL,
                        role ENUM('admin', 'customer') DEFAULT 'customer',
                        status ENUM('active', 'inactive') DEFAULT 'active',
                        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                    )",
                    
                    "CREATE TABLE IF NOT EXISTS {$prefix}customers (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        user_id INT,
                        company_name VARCHAR(255),
                        phone VARCHAR(50),
                        address TEXT,
                        city VARCHAR(100),
                        state VARCHAR(100),
                        country VARCHAR(100),
                        postal_code VARCHAR(20),
                        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                        FOREIGN KEY (user_id) REFERENCES {$prefix}users(id) ON DELETE CASCADE
                    )",
                    
                    "CREATE TABLE IF NOT EXISTS {$prefix}invoices (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        customer_id INT NOT NULL,
                        invoice_number VARCHAR(50) UNIQUE NOT NULL,
                        subtotal DECIMAL(10,2) NOT NULL DEFAULT 0.00,
                        tax_amount DECIMAL(10,2) NOT NULL DEFAULT 0.00,
                        total DECIMAL(10,2) NOT NULL,
                        currency VARCHAR(3) DEFAULT 'USD',
                        status ENUM('unpaid', 'paid', 'cancelled', 'refunded') DEFAULT 'unpaid',
                        due_date DATE NOT NULL,
                        paid_date DATETIME NULL,
                        transaction_id VARCHAR(255),
                        payment_method VARCHAR(50),
                        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                    )",
                    
                    "CREATE TABLE IF NOT EXISTS {$prefix}invoice_items (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        invoice_id INT NOT NULL,
                        description VARCHAR(255) NOT NULL,
                        quantity INT DEFAULT 1,
                        price DECIMAL(10,2) NOT NULL,
                        total DECIMAL(10,2) NOT NULL,
                        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                        FOREIGN KEY (invoice_id) REFERENCES {$prefix}invoices(id) ON DELETE CASCADE
                    )",
                    
                    "CREATE TABLE IF NOT EXISTS {$prefix}transactions (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        invoice_id INT,
                        customer_id INT NOT NULL,
                        transaction_id VARCHAR(255) UNIQUE NOT NULL,
                        gateway VARCHAR(50) NOT NULL,
                        amount DECIMAL(10,2) NOT NULL,
                        currency VARCHAR(3) DEFAULT 'USD',
                        status ENUM('pending', 'completed', 'failed', 'cancelled') DEFAULT 'pending',
                        gateway_response TEXT,
                        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                        FOREIGN KEY (invoice_id) REFERENCES {$prefix}invoices(id),
                        FOREIGN KEY (customer_id) REFERENCES {$prefix}customers(id)
                    )",
                    
                    "CREATE TABLE IF NOT EXISTS {$prefix}settings (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        name VARCHAR(255) UNIQUE NOT NULL,
                        value TEXT,
                        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                    )"
                ];
                
                foreach ($tables as $sql) {
                    $pdo->exec($sql);
                }
                
                // Save database configuration
                if (!is_dir('config')) {
                    mkdir('config', 0755, true);
                }
                
                $configContent = "<?php
return [
    'dsn' => '{$dsn}',
    'username' => '{$user}',
    'password' => '{$pass}',
    'prefix' => '{$prefix}',
    'options' => [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ]
];
?>";
                
                file_put_contents('config/database.php', $configContent);
                
                $success = 'Database configured successfully! Now create your admin account.';
                $currentStep = 2;
                
            } catch (Exception $e) {
                $error = 'Database error: ' . $e->getMessage();
            }
        }
        
    } elseif (isset($_POST['create_admin'])) {
        // Step 2: Create admin user
        $firstName = trim($_POST['admin_first_name'] ?? '');
        $lastName = trim($_POST['admin_last_name'] ?? '');
        $email = trim($_POST['admin_email'] ?? '');
        $password = $_POST['admin_password'] ?? '';
        $confirmPassword = $_POST['admin_password_confirm'] ?? '';
        
        if (empty($firstName) || empty($lastName) || empty($email) || empty($password)) {
            $error = 'Please fill in all admin account fields.';
        } elseif ($password !== $confirmPassword) {
            $error = 'Passwords do not match.';
        } elseif (strlen($password) < 8) {
            $error = 'Password must be at least 8 characters long.';
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error = 'Please enter a valid email address.';
        } else {
            try {
                $config = include 'config/database.php';
                $pdo = new PDO($config['dsn'], $config['username'], $config['password'], $config['options']);
                
                // Check if admin already exists
                $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM {$config['prefix']}users WHERE email = ? OR role = 'admin'");
                $stmt->execute([$email]);
                $result = $stmt->fetch();
                
                if ($result['count'] > 0) {
                    $error = 'Admin user already exists or email is taken.';
                } else {
                    // Create admin user
                    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                    $stmt = $pdo->prepare("INSERT INTO {$config['prefix']}users (first_name, last_name, email, password, role) VALUES (?, ?, ?, ?, 'admin')");
                    $stmt->execute([$firstName, $lastName, $email, $hashedPassword]);
                    
                    // Create basic settings
                    $settings = [
                        ['site_name', 'Billing System'],
                        ['company_name', 'Your Company'],
                        ['currency', 'USD'],
                        ['timezone', 'UTC']
                    ];
                    
                    foreach ($settings as $setting) {
                        $stmt = $pdo->prepare("INSERT IGNORE INTO {$config['prefix']}settings (name, value) VALUES (?, ?)");
                        $stmt->execute($setting);
                    }
                    
                    // Mark installation as complete
                    file_put_contents('config/installed.lock', date('Y-m-d H:i:s') . "\nAdmin: {$firstName} {$lastName} ({$email})");
                    
                    $success = 'Installation completed successfully!';
                    $currentStep = 3;
                }
                
            } catch (Exception $e) {
                $error = 'Admin creation error: ' . $e->getMessage();
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Billing System Installer</title>
    <style>
        body { 
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; 
            margin: 0; 
            padding: 20px; 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        .container { 
            max-width: 600px; 
            margin: 0 auto; 
            background: white; 
            padding: 40px; 
            border-radius: 12px; 
            box-shadow: 0 10px 30px rgba(0,0,0,0.2); 
        }
        .header {
            text-align: center;
            margin-bottom: 30px;
        }
        .header h1 {
            color: #333;
            margin-bottom: 10px;
        }
        .progress { 
            background: #e9ecef; 
            height: 8px; 
            border-radius: 4px; 
            margin: 20px 0; 
            overflow: hidden;
        }
        .progress-bar { 
            background: linear-gradient(90deg, #667eea, #764ba2); 
            height: 100%; 
            border-radius: 4px; 
            transition: width 0.3s ease; 
        }
        .step-indicator {
            display: flex;
            justify-content: space-between;
            margin-bottom: 30px;
        }
        .step {
            flex: 1;
            text-align: center;
            padding: 10px;
            border-radius: 20px;
            margin: 0 5px;
            font-size: 14px;
            font-weight: 600;
        }
        .step.active {
            background: #667eea;
            color: white;
        }
        .step.completed {
            background: #28a745;
            color: white;
        }
        .step.pending {
            background: #e9ecef;
            color: #666;
        }
        .form-group { 
            margin-bottom: 20px; 
        }
        label { 
            display: block; 
            margin-bottom: 8px; 
            font-weight: 600; 
            color: #333;
        }
        input[type="text"], input[type="email"], input[type="password"] { 
            width: 100%; 
            padding: 12px; 
            border: 2px solid #e9ecef; 
            border-radius: 6px; 
            box-sizing: border-box; 
            font-size: 16px;
            transition: border-color 0.3s ease;
        }
        input:focus {
            outline: none;
            border-color: #667eea;
        }
        button { 
            background: linear-gradient(90deg, #667eea, #764ba2); 
            color: white; 
            padding: 15px 30px; 
            border: none; 
            border-radius: 6px; 
            cursor: pointer; 
            font-size: 16px; 
            font-weight: 600;
            width: 100%;
            transition: transform 0.2s ease;
        }
        button:hover { 
            transform: translateY(-2px);
        }
        .error { 
            background: #f8d7da; 
            color: #721c24; 
            padding: 15px; 
            border-radius: 6px; 
            margin-bottom: 20px; 
            border-left: 4px solid #dc3545;
        }
        .success { 
            background: #d4edda; 
            color: #155724; 
            padding: 15px; 
            border-radius: 6px; 
            margin-bottom: 20px; 
            border-left: 4px solid #28a745;
        }
        .note {
            background: #e7f3ff;
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 20px;
            border-left: 4px solid #007bff;
            font-size: 14px;
        }
        .row {
            display: flex;
            gap: 15px;
        }
        .col {
            flex: 1;
        }
        .complete-section {
            text-align: center;
            padding: 20px;
        }
        .complete-section h2 {
            color: #28a745;
            margin-bottom: 20px;
        }
        .action-buttons {
            display: flex;
            gap: 15px;
            margin-top: 30px;
        }
        .action-buttons a {
            flex: 1;
            padding: 12px 20px;
            text-decoration: none;
            border-radius: 6px;
            text-align: center;
            font-weight: 600;
            transition: transform 0.2s ease;
        }
        .action-buttons a:hover {
            transform: translateY(-2px);
        }
        .btn-primary {
            background: #007bff;
            color: white;
        }
        .btn-success {
            background: #28a745;
            color: white;
        }
        .btn-warning {
            background: #ffc107;
            color: #212529;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🚀 Billing System Installer</h1>
            <p>Set up your complete billing and payment system</p>
        </div>
        
        <div class="step-indicator">
            <div class="step <?= $currentStep >= 1 ? ($currentStep == 1 ? 'active' : 'completed') : 'pending' ?>">
                1. Database
            </div>
            <div class="step <?= $currentStep >= 2 ? ($currentStep == 2 ? 'active' : 'completed') : 'pending' ?>">
                2. Admin User
            </div>
            <div class="step <?= $currentStep >= 3 ? 'active' : 'pending' ?>">
                3. Complete
            </div>
        </div>
        
        <div class="progress">
            <div class="progress-bar" style="width: <?= ($currentStep / 3) * 100 ?>%"></div>
        </div>
        
        <?php if ($error): ?>
            <div class="error">
                <strong>Error:</strong> <?= htmlspecialchars($error) ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="success">
                <strong>Success:</strong> <?= htmlspecialchars($success) ?>
            </div>
        <?php endif; ?>
        
        <?php if ($currentStep === 1): ?>
            <h2>Database Configuration</h2>
            <div class="note">
                <strong>Before proceeding:</strong> Create a MySQL database in your cPanel and note down the credentials.
            </div>
            
            <form method="post">
                <div class="row">
                    <div class="col">
                        <div class="form-group">
                            <label>Database Host:</label>
                            <input type="text" name="db_host" value="localhost" required>
                        </div>
                    </div>
                    <div class="col">
                        <div class="form-group">
                            <label>Database Name:</label>
                            <input type="text" name="db_name" placeholder="yourusername_billing" required>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col">
                        <div class="form-group">
                            <label>Database Username:</label>
                            <input type="text" name="db_user" placeholder="yourusername_user" required>
                        </div>
                    </div>
                    <div class="col">
                        <div class="form-group">
                            <label>Database Password:</label>
                            <input type="password" name="db_pass" placeholder="Enter password">
                        </div>
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Table Prefix:</label>
                    <input type="text" name="db_prefix" value="billing_" placeholder="billing_">
                </div>
                
                <button type="submit" name="setup_database">Setup Database & Create Tables</button>
            </form>
            
        <?php elseif ($currentStep === 2): ?>
            <h2>Create Administrator Account</h2>
            <div class="note">
                Database setup complete! Now create your admin account to manage the system.
            </div>
            
            <form method="post">
                <div class="row">
                    <div class="col">
                        <div class="form-group">
                            <label>First Name:</label>
                            <input type="text" name="admin_first_name" required>
                        </div>
                    </div>
                    <div class="col">
                        <div class="form-group">
                            <label>Last Name:</label>
                            <input type="text" name="admin_last_name" required>
                        </div>
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Email Address:</label>
                    <input type="email" name="admin_email" placeholder="admin@yourdomain.com" required>
                </div>
                
                <div class="row">
                    <div class="col">
                        <div class="form-group">
                            <label>Password:</label>
                            <input type="password" name="admin_password" placeholder="Minimum 8 characters" required minlength="8">
                        </div>
                    </div>
                    <div class="col">
                        <div class="form-group">
                            <label>Confirm Password:</label>
                            <input type="password" name="admin_password_confirm" placeholder="Confirm password" required minlength="8">
                        </div>
                    </div>
                </div>
                
                <button type="submit" name="create_admin">Create Admin Account & Complete Setup</button>
            </form>
            
        <?php else: ?>
            <div class="complete-section">
                <h2>🎉 Installation Complete!</h2>
                <p>Your billing system has been successfully installed and configured.</p>
                
                <div class="success">
                    <strong>What's been set up:</strong>
                    <ul style="text-align: left; margin: 10px 0;">
                        <li>✅ Database tables created</li>
                        <li>✅ Admin account configured</li>
                        <li>✅ Basic system settings</li>
                        <li>✅ Security configurations</li>
                    </ul>
                </div>
                
                <div class="error">
                    <strong>Important Security Steps:</strong>
                    <ol style="text-align: left; margin: 10px 0;">
                        <li>Delete this installer file immediately</li>
                        <li>Delete test files (test.php, test_login.php)</li>
                        <li>Set config folder permissions to 644</li>
                    </ol>
                </div>
                
                <div class="action-buttons">
                    <a href="test_login.php" class="btn-warning">Test Login</a>
                    <a href="index.php" class="btn-success">View Your Site</a>
                </div>
                
                <div style="margin-top: 30px; font-size: 14px; color: #666;">
                    <p>Need help? Check the documentation or contact support.</p>
                </div>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>
```

## 🔧 **How This Fixed Version Works:**

### **Key Improvements:**
1. **No redirect loops** - Uses step tracking based on completed tasks
2. **Better error handling** - Shows specific error messages
3. **Visual progress** - Clear step indicators
4. **Validation** - Prevents duplicate admin creation
5. **Completion tracking** - Uses file existence to determine current step

### **Step Flow:**
1. **Step 1:** Database setup (creates `config/database.php`)
2. **Step 2:** Admin creation (only if database exists)
3. **Step 3:** Completion (creates `config/installed.lock`)

### **What to do:**
1. **Upload** `fixed_install.php` 
2. **Upload** `test_login.php`
3. **Visit** `fixed_install.php` 
4. **Complete** both steps
5. **Test login** with `test_login.php`
6. **Delete installer** files

This should work without any redirect loops! Let me know how it goes! 🚀